/**
 * Injected into page MAIN world by Chrome (bypasses CSP).
 * Wraps whatever fetch/XHR is currently in place (including other extensions' patches).
 * Sends captured data via window.postMessage → content.js picks it up.
 */
(function () {
    const SRC = 'tg-ads-sniffer';

    // ── Wrap fetch (whatever it currently is) ──
    const prevFetch = window.fetch;
    window.fetch = async function (...args) {
        const [resource, init] = args;
        const url = typeof resource === 'string' ? resource : (resource && resource.url) || '';
        const method = (init && init.method) || (resource && resource.method) || 'GET';

        let reqBody = null;
        if (init && init.body && typeof init.body === 'string') {
            reqBody = init.body;
        }

        const reqHeaders = {};
        try {
            const h = (init && init.headers) || (resource && resource.headers);
            if (h) {
                if (h instanceof Headers) {
                    h.forEach((v, k) => { reqHeaders[k] = v; });
                } else if (typeof h === 'object') {
                    Object.assign(reqHeaders, h);
                }
            }
        } catch (e) { /* ignore */ }

        let response;
        try {
            response = await prevFetch.apply(this, args);
        } catch (err) {
            window.postMessage({
                source: SRC,
                entry: {
                    type: 'fetch', url, method,
                    requestHeaders: reqHeaders,
                    requestBody: reqBody,
                    status: 0,
                    responseBody: 'Network error: ' + err.message,
                    timestamp: new Date().toISOString()
                }
            }, '*');
            throw err;
        }

        // Clone to read body without consuming the original
        try {
            const clone = response.clone();
            clone.text().then(function (text) {
                window.postMessage({
                    source: SRC,
                    entry: {
                        type: 'fetch', url, method,
                        requestHeaders: reqHeaders,
                        requestBody: reqBody,
                        status: response.status,
                        responseBody: text,
                        timestamp: new Date().toISOString()
                    }
                }, '*');
            }).catch(function () { });
        } catch (e) { /* ignore */ }

        return response;
    };

    // ── Wrap XMLHttpRequest ──
    const prevOpen = XMLHttpRequest.prototype.open;
    const prevSend = XMLHttpRequest.prototype.send;
    const prevSetHeader = XMLHttpRequest.prototype.setRequestHeader;

    XMLHttpRequest.prototype.open = function (method, url) {
        this._sniff = { method: method, url: url, headers: {} };
        return prevOpen.apply(this, arguments);
    };

    XMLHttpRequest.prototype.setRequestHeader = function (name, value) {
        if (this._sniff) this._sniff.headers[name] = value;
        return prevSetHeader.apply(this, arguments);
    };

    XMLHttpRequest.prototype.send = function (body) {
        if (this._sniff) {
            var info = this._sniff;
            info.requestBody = typeof body === 'string' ? body : null;
            this.addEventListener('load', function () {
                window.postMessage({
                    source: SRC,
                    entry: {
                        type: 'xhr',
                        url: info.url,
                        method: info.method,
                        requestHeaders: info.headers,
                        requestBody: info.requestBody,
                        status: this.status,
                        responseBody: this.responseText,
                        timestamp: new Date().toISOString()
                    }
                }, '*');
            });
        }
        return prevSend.apply(this, arguments);
    };

    // ── Extract API hash from page ──
    // The hash appears in XHR urls like /api?hash=XXXXX
    // We also try to find it in page scripts/variables
    function extractHash() {
        // Method 1: search in all <script> tags for hash pattern
        var scripts = document.querySelectorAll('script');
        for (var i = 0; i < scripts.length; i++) {
            var text = scripts[i].textContent || '';
            var match = text.match(/[?&]hash=([a-f0-9]{10,})/);
            if (match) return match[1];
        }
        // Method 2: check if there's a global ajax setup or similar
        if (window.TG_Ads && window.TG_Ads.hash) return window.TG_Ads.hash;
        return null;
    }

    // Also intercept XHR urls to auto-detect hash
    var detectedHash = null;
    var origOpenForHash = XMLHttpRequest.prototype.open;
    XMLHttpRequest.prototype.open = function (method, url) {
        if (url && typeof url === 'string') {
            var m = url.match(/[?&]hash=([a-f0-9]{10,})/);
            if (m && !detectedHash) {
                detectedHash = m[1];
                window.postMessage({ source: 'tg-ads-hash', hash: detectedHash }, '*');
            }
        }
        return origOpenForHash.apply(this, arguments);
    };

    // Try extracting hash after DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function () {
            var h = extractHash();
            if (h) window.postMessage({ source: 'tg-ads-hash', hash: h }, '*');
        });
    } else {
        var h = extractHash();
        if (h) window.postMessage({ source: 'tg-ads-hash', hash: h }, '*');
    }

    console.log('[TG Ads Sniffer] Active — capturing fetch, XHR & hash');
})();
