// Default backend URL — used as fallback and for initial config fetch
const DEFAULT_BACKEND = 'http://sibleadpars.ru';
let backendUrl = DEFAULT_BACKEND;
let token = null;
let personalLink = null;

// ── Dynamic backend discovery ──

async function discoverBackend() {
    // Try cached URL first
    const stored = await chrome.storage.local.get({ backendUrl: null });
    if (stored.backendUrl) {
        backendUrl = stored.backendUrl;
    }

    // Fetch fresh config from server (try current backendUrl, then default)
    const urls = [backendUrl];
    if (backendUrl !== DEFAULT_BACKEND) urls.push(DEFAULT_BACKEND);

    for (const base of urls) {
        try {
            const resp = await fetch(`${base}/api/config`, { method: 'GET' });
            if (resp.ok) {
                const data = await resp.json();
                if (data.backend_url) {
                    backendUrl = data.backend_url;
                    await chrome.storage.local.set({ backendUrl: data.backend_url });
                    return;
                }
            }
        } catch (e) {
            // try next
        }
    }
    // If all fail, keep whatever we had
}

// ── Helpers ──

async function getToken() {
    if (token) return token;
    const resp = await fetch(`${backendUrl}/api/token`, { method: 'POST' });
    if (!resp.ok) throw new Error(`Сервер недоступен (${resp.status})`);
    const data = await resp.json();
    token = data.access_token;
    return token;
}

function authHeaders() {
    return { 'Authorization': `Bearer ${token}`, 'Content-Type': 'application/json' };
}

function setStatus(type, text, detail) {
    const card = document.getElementById('statusCard');
    const dot = document.getElementById('statusDot');
    const textEl = document.getElementById('statusText');
    const detailEl = document.getElementById('statusDetail');

    card.className = 'status-card ' + type;
    dot.className = 'status-dot ' + (type === 'ok' ? 'green' : type === 'no' ? 'red' : 'yellow');
    textEl.textContent = text;
    detailEl.textContent = detail || '';
}

// ── Capture Session ──

async function captureSession() {
    const btn = document.getElementById('connectBtn');
    btn.disabled = true;
    btn.innerHTML = '<span class="spinner"></span>Подключение...';

    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab || !tab.url || !tab.url.includes('ads.telegram.org')) {
            setStatus('no', 'Откройте ads.telegram.org', 'Перейдите в рекламный кабинет и попробуйте снова');
            btn.disabled = false;
            btn.textContent = 'Подключиться';
            return;
        }

        const cookies = await chrome.cookies.getAll({ domain: 'ads.telegram.org' });
        if (cookies.length === 0) {
            setStatus('no', 'Нет куки', 'Войдите в ads.telegram.org и попробуйте снова');
            btn.disabled = false;
            btn.textContent = 'Подключиться';
            return;
        }

        const cookieData = cookies.map(c => ({
            name: c.name,
            value: c.value,
            domain: c.domain,
            path: c.path,
            secure: c.secure,
            httpOnly: c.httpOnly,
        }));

        const stored = await chrome.storage.local.get({ apiHash: null });
        const apiHash = stored.apiHash;

        await getToken();
        const resp = await fetch(`${backendUrl}/api/sessions`, {
            method: 'POST',
            headers: authHeaders(),
            body: JSON.stringify({
                cookies: cookieData,
                user_agent: navigator.userAgent,
                api_hash: apiHash,
            }),
        });

        if (!resp.ok) throw new Error(`Ошибка сервера (${resp.status})`);
        const result = await resp.json();

        setStatus('ok', 'Подключено', apiHash ? `API hash: ${apiHash.substring(0, 8)}...` : 'Hash ещё не обнаружен — обновите страницу кабинета');
        btn.textContent = 'Переподключить';
        btn.disabled = false;

        // Show "Open site" button
        if (result.personal_link) {
            personalLink = result.personal_link;
            document.getElementById('openSiteBtn').style.display = 'block';
        }

    } catch (e) {
        setStatus('no', 'Ошибка подключения', e.message);
        btn.disabled = false;
        btn.textContent = 'Подключиться';
    }
}

document.getElementById('connectBtn').addEventListener('click', captureSession);

document.getElementById('openSiteBtn').addEventListener('click', () => {
    if (personalLink) {
        chrome.tabs.create({ url: personalLink });
    }
});

// ── Auto-capture on popup open ──

(async () => {
    // Discover backend URL first
    await discoverBackend();

    try {
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (tab && tab.url && tab.url.includes('ads.telegram.org')) {
            await captureSession();
        } else {
            setStatus('wait', 'Ожидание подключения', 'Откройте ads.telegram.org');
        }
    } catch (e) {
        // ignore
    }
})();
